package omq.server.remote.request;

import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;
import java.util.Vector;

import omq.Remote;
import omq.common.broker.Broker;
import omq.common.event.EventListener;
import omq.common.event.EventTrigger;
import omq.common.message.response.DefaultResponse;
import omq.common.message.response.ExceptionResponse;
import omq.common.message.response.Response;
import omq.common.util.ParameterQueue;
import omq.exception.SerializerException;

import com.rabbitmq.client.Channel;
import com.rabbitmq.client.Connection;
import com.rabbitmq.client.ConsumerCancelledException;
import com.rabbitmq.client.QueueingConsumer;
import com.rabbitmq.client.QueueingConsumer.Delivery;
import com.rabbitmq.client.ShutdownSignalException;

/**
 * 
 * @author Sergi Toda <sergi.toda@estudiants.urv.cat>
 * 
 */
public abstract class RemoteObject extends Thread implements Remote {

	private static final long serialVersionUID = -1778953938739846450L;

	private String UID;
	private RemoteWrapper remoteWrapper;
	private Connection connection;
	private Channel channel;
	private QueueingConsumer consumer;
	private boolean killed = false;

	private static final Map<String, Class<?>> primitiveClasses = new HashMap<String, Class<?>>();

	static {
		primitiveClasses.put("byte", Byte.class);
		primitiveClasses.put("short", Short.class);
		primitiveClasses.put("char", Character.class);
		primitiveClasses.put("int", Integer.class);
		primitiveClasses.put("long", Long.class);
		primitiveClasses.put("float", Float.class);
		primitiveClasses.put("double", Double.class);
	}

	public void start(String reference, Properties env) throws Exception {
		this.UID = reference;

		// Get num threads to use
		int numThreads = Integer.parseInt(env.getProperty(ParameterQueue.NUM_THREADS));
		remoteWrapper = new RemoteWrapper(this, numThreads);

		// Get info about which exchange and queue will use
		String exchange = env.getProperty(ParameterQueue.RPC_EXCHANGE);
		String queue = UID;
		String routingKey = UID;

		// Start connection and channel
		connection = Broker.getConnection();
		channel = connection.createChannel();

		// Declares and bindings
		channel.exchangeDeclare(exchange, "direct");
		channel.queueDeclare(queue, false, false, false, null);
		channel.queueBind(queue, exchange, routingKey);

		// Declare the event topic fanout
		channel.exchangeDeclare(UID, "fanout");

		// Declare a new consumer
		consumer = new QueueingConsumer(channel);
		channel.basicConsume(queue, true, consumer);

		// Start this listener
		this.start();
	}

	@Override
	public void run() {
		while (!killed) {
			try {
				Delivery delivery = consumer.nextDelivery();
				remoteWrapper.notifyDelivery(delivery);
			} catch (InterruptedException i) {
				i.printStackTrace();
			} catch (ShutdownSignalException e) {
				e.printStackTrace();
			} catch (ConsumerCancelledException e) {
				e.printStackTrace();
			} catch (SerializerException e) {
				e.printStackTrace();
			} catch (Exception e) {
				e.printStackTrace();
			}
		}
	}

	@Override
	public String getRef() {
		return UID;
	}

	public void kill() throws IOException {
		interrupt();
		killed = true;
		channel.close();
		connection.close();
		remoteWrapper.stopRemoteWrapper();
	}

	@Override
	public Response invokeMethod(String methodName, Vector<Object> args) throws Exception {
		Object[] arguments = new Object[args.size()];

		for (int i = 0; i < args.size(); i++) {
			Object arg = args.get(i);
			// TODO: what happens if the object is a remote object?
			arguments[i] = arg;
		}

		// Get the specific method identified by methodName and its arguments
		Method method = loadMethod(methodName, args);

		try {
			Object result = method.invoke(this, arguments);
			return new DefaultResponse(this.getRef(), result);
		} catch (InvocationTargetException e) {
			return new ExceptionResponse(this.getRef(), e.getTargetException());
		}
	}

	private Method loadMethod(String methodName, Vector<Object> args) throws NoSuchMethodException {
		Method m = null;

		// Obtain the class reference
		Class<?> clazz = this.getClass();
		Class<?>[] argArray = new Class<?>[args.size()];

		for (int i = 0; i < args.size(); i++) {
			argArray[i] = args.get(i).getClass();
		}

		try {
			m = clazz.getMethod(methodName, argArray);
		} catch (NoSuchMethodException nsm) {
			m = loadMethodWithPrimitives(methodName, argArray);
		}
		return m;
	}

	private Method loadMethodWithPrimitives(String methodName, Class<?>[] argArray) throws NoSuchMethodException {
		Method[] methods = this.getClass().getMethods();
		int length = argArray.length;

		for (Method method : methods) {
			String name = method.getName();
			int argsLength = method.getParameterTypes().length;

			if (name.equals(methodName) && length == argsLength) {
				// This array can have primitive types inside
				Class<?>[] params = method.getParameterTypes();

				boolean found = true;

				for (int i = 0; i < length; i++) {
					if (params[i].isPrimitive()) {
						Class<?> paramWrapper = primitiveClasses.get(params[i].getName());

						if (!paramWrapper.equals(argArray[i])) {
							found = false;
							break;
						}
					}
				}
				if (found) {
					return method;
				}
			}
		}
		throw new NoSuchMethodException(methodName);
	}

	@Override
	public void notify(Object obj) throws Exception {
		EventTrigger.publish(UID, obj);
	}

	public void addListener(EventListener eventListener) throws Exception {

	}

	public void removeListener(EventListener eventListener) throws Exception {

	}

	public Collection<EventListener> getListeners() throws Exception {
		return null;
	}

}
