package omq.server.remote.request;

import java.io.IOException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import omq.Remote;
import omq.common.broker.Broker;
import omq.common.util.ParameterQueue;
import omq.exception.SerializerException;

import com.rabbitmq.client.Channel;
import com.rabbitmq.client.Connection;
import com.rabbitmq.client.ConsumerCancelledException;
import com.rabbitmq.client.QueueingConsumer;
import com.rabbitmq.client.QueueingConsumer.Delivery;
import com.rabbitmq.client.ShutdownSignalException;

/**
 * 
 * @author Sergi Toda <sergi.toda@estudiants.urv.cat>
 * 
 */
public abstract class RemoteObject extends Thread implements Remote {

	private static final long serialVersionUID = -1778953938739846450L;

	private String UID;
	private transient RemoteWrapper remoteWrapper;
	private transient Map<String, List<Class<?>>> params;
	private transient Connection connection;
	private transient Channel channel;
	private transient QueueingConsumer consumer;
	private transient boolean killed = false;

	private static final Map<String, Class<?>> primitiveClasses = new HashMap<String, Class<?>>();

	static {
		primitiveClasses.put("byte", Byte.class);
		primitiveClasses.put("short", Short.class);
		primitiveClasses.put("char", Character.class);
		primitiveClasses.put("int", Integer.class);
		primitiveClasses.put("long", Long.class);
		primitiveClasses.put("float", Float.class);
		primitiveClasses.put("double", Double.class);
	}

	public RemoteObject() {
		params = new HashMap<String, List<Class<?>>>();
		for (Method m : this.getClass().getMethods()) {
			List<Class<?>> list = new ArrayList<Class<?>>();
			for (Class<?> clazz : m.getParameterTypes()) {
				list.add(clazz);
			}
			params.put(m.getName(), list);
		}
	}

	public void start(String reference, Properties env) throws Exception {
		this.UID = reference;

		// Get num threads to use
		int numThreads = 4;// Integer.parseInt(env.getProperty(ParameterQueue.NUM_THREADS));
		remoteWrapper = new RemoteWrapper(this, numThreads);

		// Get info about which exchange and queue will use
		String exchange = env.getProperty(ParameterQueue.RPC_EXCHANGE);
		String queue = UID;
		String routingKey = UID;

		// Start connection and channel
		connection = Broker.getConnection();
		channel = connection.createChannel();

		// Declares and bindings
		channel.exchangeDeclare(exchange, "direct");
		channel.queueDeclare(queue, false, false, false, null);
		channel.queueBind(queue, exchange, routingKey);

		// Declare the event topic fanout
		channel.exchangeDeclare(UID, "fanout");

		// Declare a new consumer
		consumer = new QueueingConsumer(channel);
		channel.basicConsume(queue, true, consumer);

		// Start this listener
		this.start();
	}

	@Override
	public void run() {
		while (!killed) {
			try {
				Delivery delivery = consumer.nextDelivery();
				remoteWrapper.notifyDelivery(delivery);
			} catch (InterruptedException i) {
				i.printStackTrace();
			} catch (ShutdownSignalException e) {
				e.printStackTrace();
			} catch (ConsumerCancelledException e) {
				e.printStackTrace();
			} catch (SerializerException e) {
				e.printStackTrace();
			} catch (Exception e) {
				e.printStackTrace();
			}
		}
	}

	@Override
	public String getRef() {
		return UID;
	}

	public void kill() throws IOException {
		interrupt();
		killed = true;
		channel.close();
		connection.close();
		remoteWrapper.stopRemoteWrapper();
	}

	public Object invokeMethod(String methodName,Object[] arguments) throws Exception {

		// Get the specific method identified by methodName and its arguments
		Method method = loadMethod(methodName, arguments);

		return method.invoke(this, arguments);
	}

	private Method loadMethod(String methodName,Object[] args) throws NoSuchMethodException {
		Method m = null;

		// Obtain the class reference
		Class<?> clazz = this.getClass();
		Class<?>[] argArray = new Class<?>[args.length];

		for (int i = 0; i < args.length; i++) {
			argArray[i] = args[i].getClass();
		}

		try {
			m = clazz.getMethod(methodName, argArray);
		} catch (NoSuchMethodException nsm) {
			m = loadMethodWithPrimitives(methodName, argArray);
		}
		return m;
	}

	private Method loadMethodWithPrimitives(String methodName, Class<?>[] argArray) throws NoSuchMethodException {
		Method[] methods = this.getClass().getMethods();
		int length = argArray.length;

		for (Method method : methods) {
			String name = method.getName();
			int argsLength = method.getParameterTypes().length;

			if (name.equals(methodName) && length == argsLength) {
				// This array can have primitive types inside
				Class<?>[] params = method.getParameterTypes();

				boolean found = true;

				for (int i = 0; i < length; i++) {
					if (params[i].isPrimitive()) {
						Class<?> paramWrapper = primitiveClasses.get(params[i].getName());

						if (!paramWrapper.equals(argArray[i])) {
							found = false;
							break;
						}
					}
				}
				if (found) {
					return method;
				}
			}
		}
		throw new NoSuchMethodException(methodName);
	}

	public List<Class<?>> getParams(String methodName) {
		return params.get(methodName);
	}

	public Channel getChannel() {
		return channel;
	}

}
